<?php

/**
 * Custom Recent Comments widget.
 * 
 * Resources:
 * https://lonewolfonline.net/creating-wordpress-widgets/
 * https://www.hostinger.com/tutorials/how-to-create-custom-widget-in-wordpress
 */
class Custom_Recent_Comments_Widget extends WP_Widget_Recent_Comments {

    public function widget( $args, $instance ) {
		static $first_instance = true;

		if ( ! isset( $args['widget_id'] ) ) {
			$args['widget_id'] = $this->id;
		}

		$output = '';

		$default_title = __( 'Recent Comments' );
		$title         = ( ! empty( $instance['title'] ) ) ? $instance['title'] : $default_title;

		/** This filter is documented in wp-includes/widgets/class-wp-widget-pages.php */
		$title = apply_filters( 'widget_title', $title, $instance, $this->id_base );

		$number = ( ! empty( $instance['number'] ) ) ? absint( $instance['number'] ) : 5;
		if ( ! $number ) {
			$number = 5;
		}

		if ( 'aeiforia-project' === get_post_type() ) {
			$post_type = 'aeiforia-project';
		}
		else if ( 'eruahaus-project' === get_post_type() ) {
			$post_type = 'eruahaus-project';
		}
		else if ( 'showroom' === get_post_type() ) {
			$post_type = 'showroom';
		}
		else if ( 'creative-blog-post' === get_post_type() ) {
			$post_type = 'creative-blog-post';
		}
		else if ( 'hackathon' === get_post_type() ) {
			$post_type = 'hackathon';
		}
		else if ( 'funding-opportunity' === get_post_type() ) {
			$post_type = 'funding-opportunity';
		}
		else if ( 'job-opportunity' === get_post_type() ) {
			$post_type = 'job-opportunity';
		}
		else {
			$post_type = 'post';
		}

		$comments = get_comments(
			/**
			 * Filters the arguments for the Recent Comments widget.
			 *
			 * @since 3.4.0
			 * @since 4.9.0 Added the `$instance` parameter.
			 *
			 * @see WP_Comment_Query::query() for information on accepted arguments.
			 *
			 * @param array $comment_args An array of arguments used to retrieve the recent comments.
			 * @param array $instance     Array of settings for the current widget.
			 */
			apply_filters(
				'widget_comments_args',
				array(
					'number'      => $number,
					'status'      => 'approve',
					'post_status' => 'publish',
                    'post_type'   => $post_type,
				),
				$instance
			)
		);

		$output .= $args['before_widget'];
		if ( $title ) {
			$output .= $args['before_title'] . $title . $args['after_title'];
		}

		$recent_comments_id = ( $first_instance ) ? 'recentcomments' : "recentcomments-{$this->number}";
		$first_instance     = false;

		$format = current_theme_supports( 'html5', 'navigation-widgets' ) ? 'html5' : 'xhtml';

		/** This filter is documented in wp-includes/widgets/class-wp-nav-menu-widget.php */
		$format = apply_filters( 'navigation_widgets_format', $format );

		if ( 'html5' === $format ) {
			// The title may be filtered: Strip out HTML and make sure the aria-label is never empty.
			$title      = trim( strip_tags( $title ) );
			$aria_label = $title ? $title : $default_title;
			$output    .= '<nav aria-label="' . esc_attr( $aria_label ) . '">';
		}

		$output .= '<ul id="' . esc_attr( $recent_comments_id ) . '">';
		if ( is_array( $comments ) && $comments ) {
			// Prime cache for associated posts. (Prime post term cache if we need it for permalinks.)
			$post_ids = array_unique( wp_list_pluck( $comments, 'comment_post_ID' ) );
			_prime_post_caches( $post_ids, strpos( get_option( 'permalink_structure' ), '%category%' ), false );

			foreach ( (array) $comments as $comment ) {
				$output .= '<li class="recentcomments">';
				$output .= sprintf(
					/* translators: Comments widget. 1: Comment author, 2: Post link. */
					_x( '%1$s on %2$s', 'widgets' ),
					'<span class="comment-author-link">' . get_comment_author_link( $comment ) . '</span>',
					'<a href="' . esc_url( get_comment_link( $comment ) ) . '">' . get_the_title( $comment->comment_post_ID ) . '</a>'
				);
				$output .= '</li>';
			}
		}
		$output .= '</ul>';

		if ( 'html5' === $format ) {
			$output .= '</nav>';
		}

		$output .= $args['after_widget'];

		echo $output;
	}

}
