<?php
/**
 * This file contains various theme configurations.
 * Setup of styles (.css), scripts (.js), languages, features, menus, widgets etc.
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit; // Exit if accessed directly.
}

//================================================================================
// THEME TWEAKS - GENERAL
//================================================================================

/**
 * Disable Admin Bar for All Users Except for Administrators.
 * Administrators can still disable/enable it in their own profile options.
 */
function erua_remove_admin_bar() {
    if ( !current_user_can('administrator') && !is_admin() ) {
        show_admin_bar(false);
    }
}
add_action('after_setup_theme', 'erua_remove_admin_bar');

/**
 * Enable UX Builder for custom post types.
 * 
 * https://docs.uxthemes.com/article/221-how-to-enable-ux-builder-for-custom-post-types
 * https://wordpress.stackexchange.com/questions/374033/how-to-enable-ux-builder-for-custom-post-types-in-flatsome
 */
function erua_add_ux_builder_to_cpt() {
	if ( function_exists( 'add_ux_builder_post_type' ) ) {
        add_ux_builder_post_type( 'document' );
    }
}
//add_action( 'init', 'erua_add_ux_builder_to_cpt' );

/**
 * The flatsome_lightbox_close_button filter
 * allows changing the lightbox close button markup.
 * 
 * https://docs.uxthemes.com/article/378-lightbox-close-button
 */
add_filter( 'flatsome_lightbox_close_button', function ( $html ) {
	$html = '<button title="'. __('Close (Esc)', 'erua') . '" type="button" class="mfp-close">';
	/* f.ex. We're replacing the SVG icon into another one. */
	$html .= '<svg xmlns="http://www.w3.org/2000/svg" width="28" height="28" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" class="feather feather-x"><line x1="18" y1="6" x2="6" y2="18"></line><line x1="6" y1="6" x2="18" y2="18"></line></svg>';
	$html .= '</button>';

	return $html;
} );

/**
 * Display navigation to next/previous pages when applicable
 */
function flatsome_content_nav( $nav_id ) {
    global $wp_query, $post;

    $post_type = $post->post_type;
    $taxonomy = 'category';
    $in_same_term = false;

    switch ($post_type) {
        case 'aeiforia-project':
            $taxonomy = 'aeiforia-project-category';
            $in_same_term = true;
            break;
        case 'eruahaus-project':
            $taxonomy = 'eruahaus-project-focus-area';
            $in_same_term = true;
            break;
        case 'showroom':
            $taxonomy = 'showroom-category';
            $in_same_term = true;
            break;
    }

    // Don't print empty markup on single pages if there's nowhere to navigate.
    if ( is_single() ) {
        $previous = ( is_attachment() ) ? get_post( $post->post_parent ) : get_adjacent_post( $in_same_term, '', true, $taxonomy );
        $next = get_adjacent_post( $in_same_term, '', false, $taxonomy );

        if ( ! $next && ! $previous )
            return;
    }

    // Don't print empty markup in archives if there's only one page.
    if ( $wp_query->max_num_pages < 2 && ( is_home() || is_archive() || is_search() ) )
        return;

    $nav_class = ( is_single() ) ? 'navigation-post' : 'navigation-paging';

    ?>
    <?php if ( is_single() ) : // navigation links for single posts ?>
    <nav role="navigation" id="<?php echo esc_attr( $nav_id ); ?>" class="<?php echo $nav_class; ?>">
    <div class="flex-row next-prev-nav bt bb">
        <div class="flex-col flex-grow nav-prev text-left">
                <?php previous_post_link( '<div class="nav-previous">%link</div>','<span class="hide-for-small">' .get_flatsome_icon('icon-angle-left') . _x( '', 'Previous post link', 'flatsome' ) . '</span> %title', $in_same_term, '', $taxonomy ); ?>

        </div>
        <div class="flex-col flex-grow nav-next text-right">
                <?php next_post_link( '<div class="nav-next">%link</div>', '%title <span class="hide-for-small">'. get_flatsome_icon('icon-angle-right') . _x( '', 'Next post link', 'flatsome' ) . '</span>', $in_same_term, '', $taxonomy ); ?>
        </div>
    </div>

    <?php elseif ( $wp_query->max_num_pages > 1 && ( is_home() || is_archive() || is_search() ) ) : // navigation links for home, archive, and search pages ?>

    <div class="flex-row">
        <div class="flex-col flex-grow">
<?php if ( get_next_posts_link() ) : ?>
        <div class="nav-previous"><?php next_posts_link( __( '<span class="icon-angle-left"></span> Older posts', 'flatsome' ) ); ?></div>
        <?php endif; ?>
        </div>
        <div class="flex-col flex-grow">
        <?php if ( get_previous_posts_link() ) : ?>
            <div class="nav-next"><?php previous_posts_link( __( 'Newer posts <span class="icon-angle-right"></span>', 'flatsome' ) ); ?></div>
        <?php endif; ?>		</div>
    </div>
    <?php endif; ?>
    </nav>

    <?php
}

/**
 * Prints HTML with meta information for the current post-date/time and author.
 *
 * Rewrite the post-date/time and author.
 */
function flatsome_posted_on() {
    $time_string = '<time class="entry-date published updated" datetime="%1$s">%2$s</time>';
    if ( get_the_time( 'U' ) !== get_the_modified_time( 'U' ) ) {
        $time_string = '<time class="entry-date published" datetime="%1$s">%2$s</time><time class="updated" datetime="%3$s">%4$s</time>';
    }

    $time_string = sprintf( $time_string,
        esc_attr( get_the_date( 'c' ) ),
        esc_html( get_the_date( 'j F Y, H:i' ) ),
        esc_attr( get_the_modified_date( 'c' ) ),
        esc_html( get_the_modified_date( 'j F Y, H:i' ) )
    );

    $posted_on = sprintf(
        esc_html_x( 'Posted on %s', 'post date', 'erua' ),
        '<a href="' . esc_url( get_permalink() ) . '" rel="bookmark">' . $time_string . '</a>'
    );

    $byline = sprintf(
        esc_html_x( 'by %s', 'post author', 'erua' ),
        '<span class="meta-author vcard"><a class="url fn n" href="' . esc_url( get_author_posts_url( get_the_author_meta( 'ID' ) ) ) . '">' . esc_html( get_the_author() ) . '</a></span>'
    );

    echo '<span class="posted-on">' . $posted_on . '</span><span class="byline"> ' . $byline . '</span>';

}

/**
 * - "Default featured image" plugin -
 * 
 * Set a different default featured image for custom post types. 
 *
 * https://wordpress.org/plugins/default-featured-image/
 */
function dfi_cpt( $dfi_id, $post_id ) {
    $post = get_post( $post_id );

    if ( 'aeiforia-project' === $post->post_type ) {
        global $PROJECT_DEFAULT_THUMBNAIL_IMAGE;
        return $PROJECT_DEFAULT_THUMBNAIL_IMAGE; # the image id
    }

    // if ( 'some-cpt-name' === $post->post_type ) {
    //     return null;
    // }

    return $dfi_id; // the original featured image id.
}
//add_filter( 'dfi_thumbnail_id', 'dfi_cpt', 10, 2 );

//================================================================================
// THEME TWEAKS - REWRITING THE LIVE AJAX SEARCH
//================================================================================

/**
 * Remove parent theme's AJAX search function.
 *
 * @return void
 */
function erua_remove_parent_actions() {
    remove_action( 'wp_ajax_flatsome_ajax_search_products', 'flatsome_ajax_search' );
    remove_action( 'wp_ajax_nopriv_flatsome_ajax_search_products', 'flatsome_ajax_search' );
}
add_action( 'wp_loaded', 'erua_remove_parent_actions' );

/**
 * Search AJAX handler.
 */
function erua_ajax_search() {
	// The string from search text field.
	$query        = apply_filters( 'flatsome_ajax_search_query', $_REQUEST['query'] );
	$wc_activated = is_woocommerce_activated();
	$products     = array();
	$posts        = array();
	$sku_products = array();
	$tag_products = array();
	$suggestions  = array();

    // If the search form has a hidden input to restrict the search by specific post type:
    $is_specific_search = null;
    $is_specific_search_for_posts = ( isset($_REQUEST['post_type']) && $_REQUEST['post_type'] === 'post' ) ? true : false;
    $is_specific_search_for_aeiforia_projects = ( isset($_REQUEST['post_type']) && $_REQUEST['post_type'] === 'aeiforia-project' ) ? true : false;
    $is_specific_search_for_eruahaus_projects = ( isset($_REQUEST['post_type']) && $_REQUEST['post_type'] === 'eruahaus-project' ) ? true : false;
    $is_specific_search_for_showroom_presentations = ( isset($_REQUEST['post_type']) && $_REQUEST['post_type'] === 'showroom' ) ? true : false;
    $is_specific_search_for_creative_blog_posts = ( isset($_REQUEST['post_type']) && $_REQUEST['post_type'] === 'creative-blog-post' ) ? true : false;
    $is_specific_search_for_hackathons = ( isset($_REQUEST['post_type']) && $_REQUEST['post_type'] === 'hackathon' ) ? true : false;
    $is_specific_search_for_funding_opportunities = ( isset($_REQUEST['post_type']) && $_REQUEST['post_type'] === 'funding-opportunity' ) ? true : false;
    $is_specific_search_for_job_opportunities = ( isset($_REQUEST['post_type']) && $_REQUEST['post_type'] === 'job-opportunity' ) ? true : false;
    $is_specific_search_for_events = ( isset($_REQUEST['post_type']) && $_REQUEST['post_type'] === 'event' ) ? true : false;

    /**
     * https://thaibinhweb.net/sua-loi-file-tim-kiem-search-cua-flatsome-bi-treo-khi-tim-kiem/
     * https://docs.uxthemes.com/article/385-hooks#flatsome_ajax_search_function
     * https://wordpress.org/support/topic/partial-searches-not-working/
     */
    if ( $is_specific_search_for_posts ) {
        add_filter( 'flatsome_ajax_search_post_type', function( $post_type ) { return array('post'); } );
    }
    else if ( $is_specific_search_for_aeiforia_projects ) {
        add_filter( 'flatsome_ajax_search_post_type', function( $post_type ) { return array('aeiforia-project'); } );
    }
    else if ( $is_specific_search_for_eruahaus_projects ) {
        add_filter( 'flatsome_ajax_search_post_type', function( $post_type ) { return array('eruahaus-project'); } );
    }
    else if ( $is_specific_search_for_showroom_presentations ) {
        add_filter( 'flatsome_ajax_search_post_type', function( $post_type ) { return array('showroom'); } );
    }
    else if ( $is_specific_search_for_creative_blog_posts ) {
        add_filter( 'flatsome_ajax_search_post_type', function( $post_type ) { return array('creative-blog-post'); } );
    }
    else if ( $is_specific_search_for_hackathons ) {
        add_filter( 'flatsome_ajax_search_post_type', function( $post_type ) { return array('hackathon'); } );
    }
    else if ( $is_specific_search_for_funding_opportunities ) {
        add_filter( 'flatsome_ajax_search_post_type', function( $post_type ) { return array('funding-opportunity'); } );
    }
    else if ( $is_specific_search_for_job_opportunities ) {
        add_filter( 'flatsome_ajax_search_post_type', function( $post_type ) { return array('job-opportunity'); } );
    }
    else if ( $is_specific_search_for_events ) {
        add_filter( 'flatsome_ajax_search_post_type', function( $post_type ) { return array('event'); } );
    }
    else {
        $is_specific_search = false;
        // THESE ARE ALL THE DEFINED POST TYPES FOR THE AJAX SEARCH:
        //add_filter( 'flatsome_ajax_search_post_type', function( $post_type ) { return array('page', 'post', 'product', 'document'); } );
        add_filter( 'flatsome_ajax_search_post_type', function( $post_type ) {
            return array('page', 'post', 'aeiforia-project', 'eruahaus-project', 'showroom', 'creative-blog-post', 'hackathon', 'funding-opportunity', 'job-opportunity', 'event');
        } );
    }

	$args = array(
		's'                   => $query,
		'orderby'             => '',
		//'post_type'           => ( !empty($_POST) && isset($_POST['post_type']) && $_POST['post_type'] === 'post' ) ? array('post') : array(),
        //'post_type'           => array('page', 'post', 'product', 'document'),
        'post_type'           => array(),
		'post_status'         => 'publish',
        //'post__not_in'        => [1221, 1472, 1322, 1052, 2038, 564, 566],
		'posts_per_page'      => 20, # Default: 100
		'ignore_sticky_posts' => 1,
		'post_password'       => '',
		'suppress_filters'    => false,
	);

    if ( $excluded_pages_array = get_field('search_excluded_pages', 'option') ) {
        $args['post__not_in'] = $excluded_pages_array;
    }

	if ( $wc_activated && ! $is_specific_search ) {
		$products     = flatsome_ajax_search_get_products( 'product', $args );
		$sku_products = get_theme_mod( 'search_by_sku', 0 ) ? flatsome_ajax_search_get_products( 'sku', $args ) : array();
		$tag_products = get_theme_mod( 'search_by_product_tag', 0 ) ? flatsome_ajax_search_get_products( 'tag', $args ) : array();
	}

	if ( ( ! $wc_activated || get_theme_mod( 'search_result', 1 ) ) && ! isset( $_REQUEST['product_cat'] ) ) {
		$posts = flatsome_ajax_search_posts( $args );
	}

	$results = array_merge( $products, $sku_products, $tag_products, $posts );

	foreach ( $results as $key => $post ) {
		if ( $wc_activated && ( $post->post_type === 'product' || $post->post_type === 'product_variation' ) ) {
			$product = wc_get_product( $post );

			if ( $product->get_parent_id() ) {
				$parent_product = wc_get_product( $product->get_parent_id() );
				$visible        = $parent_product->get_catalog_visibility() === 'visible' || $parent_product->get_catalog_visibility() === 'search';
				if ( $parent_product->get_status() !== 'publish' || ! $visible ) {
					unset( $results[ $key ] );
					continue;
				}
			}

			$product_image = wp_get_attachment_image_src( get_post_thumbnail_id( $product->get_id() ) );

			$suggestions[] = array(
				'type'  => 'Product',
				'id'    => $product->get_id(),
				'value' => $product->get_title(),
				'url'   => $product->get_permalink(),
				'img'   => $product_image ? $product_image[0] : '',
				'price' => $product->get_price_html(),
			);
		} else {
			$suggestions[] = array(
				'type'  => 'Page',
				'id'    => $post->ID,
				'value' => get_the_title( $post->ID ),
				'url'   => get_the_permalink( $post->ID ),
				'img'   => get_the_post_thumbnail_url( $post->ID, 'thumbnail' ),
				'price' => '',
			);
		}
	}

	if ( empty( $results ) ) {

        if ( $is_specific_search_for_posts ) {
            $no_results = $wc_activated ? __( 'No results', 'erua' ) : __( 'No matches found', 'erua' );
        }
        else if ( $is_specific_search_for_aeiforia_projects ) {
            $no_results = $wc_activated ? __( 'No results', 'erua' ) : __( 'No matches found', 'erua' );
        }
        else {
            //$no_results = $wc_activated ? __( 'No products found.', 'erua' ) . json_encode($_REQUEST) . var_export($is_specific_search, true) : __( 'No matches found', 'erua' );
            $no_results = $wc_activated ? __( 'No matches found.', 'erua' ) : __( 'No matches found', 'erua' );
        }

		$suggestions[] = array(
			'id'    => -1,
			'value' => $no_results,
			'url'   => '',
		);
	}

	$suggestions = flatsome_unique_suggestions( array( $products, $sku_products, $tag_products ), $suggestions );

    // Diagnose the Post Type
    // $suggestions[] = array(
    //     'id'    => -1,
    //     'value' => 'Post type: ' . $_REQUEST['post_type'],
    //     'url'   => '',
    // );

	wp_send_json( array( 'suggestions' => $suggestions ) );
}
add_action( 'wp_ajax_flatsome_ajax_search_products', 'erua_ajax_search' );
add_action( 'wp_ajax_nopriv_flatsome_ajax_search_products', 'erua_ajax_search' );

//================================================================================
// THEME TWEAKS - REWRITING THE SEARCH RESULTS PAGE
//================================================================================

/**
 * Remove parent theme's Search Results.
 *
 * @return void
 */
function erua_remove_parent_actions2() {
    remove_action( 'woocommerce_after_main_content', 'erua_pages_in_search_results' );
}
//add_action( 'wp_loaded', 'erua_remove_parent_actions2' );

// Add Pages and blog posts to top of search results if set.
function erua_pages_in_search_results(){
    if(!is_search() || !get_theme_mod('search_result', 1)) return;
    global $post;
    ?>
    <?php if( get_search_query() ) : ?>
    <?php

    //remove_query_arg('post_type');

    /**
     * Include pages and posts in search
     */
    query_posts( array( 'post_type' => array( 'post'), 's' => get_search_query() ) );

    $posts = array();
    while ( have_posts() ) : the_post();
    array_push($posts, $post->ID);
    endwhile;

    wp_reset_query();

    // Get pages
    query_posts( array( 'post_type' => array( 'page'), 's' => get_search_query() ) );
    $pages = array();
    while ( have_posts() ) : the_post();
    $wc_page = false;
    if($post->post_type == 'page'){
        foreach (array('shop', 'cart', 'checkout', 'view_order', 'terms') as $wc_page_type) {
        if( $post->ID == wc_get_page_id($wc_page_type) ) $wc_page = true;
        }
    }
    if( !$wc_page ) array_push($pages, $post->ID);
    endwhile;

    wp_reset_query();

    /**
     * Include Documents in search
     */
    query_posts( array( 'post_type' => array( 'document'), 's' => get_search_query() ) );
    $documents = array();
    while ( have_posts() ) : the_post();
    $wc_page = false;
    if($post->post_type == 'document'){
        foreach (array('shop', 'cart', 'checkout', 'view_order', 'terms') as $wc_page_type) {
            if( $post->ID == wc_get_page_id($wc_page_type) ) $wc_page = true;
        }
    }
    if( !$wc_page ) array_push($documents, $post->ID);
    endwhile;

    wp_reset_query();

    /**
     * Include Case Studies in search
     */
    query_posts( array( 'post_type' => array( 'case-study'), 's' => get_search_query() ) );
    $case_studies = array();
    while ( have_posts() ) : the_post();
    $wc_page = false;
    if($post->post_type == 'case-study'){
        foreach (array('shop', 'cart', 'checkout', 'view_order', 'terms') as $wc_page_type) {
            if( $post->ID == wc_get_page_id($wc_page_type) ) $wc_page = true;
        }
    }
    if( !$wc_page ) array_push($case_studies, $post->ID);
    endwhile;

    /**
     * Include WFRM Solutions in search
     */
    query_posts( array( 'post_type' => array( 'wfrm-solution'), 's' => get_search_query() ) );
    $wfrm_solutions = array();
    while ( have_posts() ) : the_post();
    $wc_page = false;
    if($post->post_type == 'wfrm-solution'){
        foreach (array('shop', 'cart', 'checkout', 'view_order', 'terms') as $wc_page_type) {
            if( $post->ID == wc_get_page_id($wc_page_type) ) $wc_page = true;
        }
    }
    if( !$wc_page ) array_push($wfrm_solutions, $post->ID);
    endwhile;

    wp_reset_query();

    /**
     * Include TechMall Solutions in search
     */
    query_posts( array( 'post_type' => array( 'techmall-solution'), 's' => get_search_query() ) );
    $techmall_solutions = array();
    while ( have_posts() ) : the_post();
    $wc_page = false;
    if($post->post_type == 'techmall-solution'){
        foreach (array('shop', 'cart', 'checkout', 'view_order', 'terms') as $wc_page_type) {
            if( $post->ID == wc_get_page_id($wc_page_type) ) $wc_page = true;
        }
    }
    if( !$wc_page ) array_push($techmall_solutions, $post->ID);
    endwhile;

    wp_reset_query();

    //if ($search_query != null && strlen($search_query) >= 3 && have_posts()) {}
    global $search_time;
    global $search_query;
    $posts_count = ( !empty( $posts ) ) ? count($posts) : 0;
    $pages_count = ( !empty( $pages ) ) ? count($pages) : 0;
    $documents_count = ( !empty( $documents ) ) ? count($documents) : 0;
    $case_studies_count = ( !empty( $case_studies ) ) ? count($case_studies) : 0;
    $wfrm_solutions_count = ( !empty( $wfrm_solutions ) ) ? count($wfrm_solutions) : 0;
    $techmall_solutions_count = ( !empty( $techmall_solutions ) ) ? count($techmall_solutions) : 0;

    echo '<div id="search-aftermath">' . __( 'Your search for', 'erua' ) . ' <span style="font-weight: bold;">' . $search_query . '</span> ' . __( 'returned', 'erua' ) . ' <span style="font-weight: bold;">' . ($posts_count+$pages_count+$documents_count+$case_studies_count+$wfrm_solutions_count+$techmall_solutions_count) . '</span> ' . __( 'results in', 'erua' ) . ' ' . $search_time . ' ' . __( 'seconds', 'erua' ) . '.</div>';

    if ( ! empty( $posts ) ) {
        echo '<hr/><h4 class="uppercase">' . esc_html__( 'Posts found', 'erua' ) . ' (' . $posts_count . ')</h4>';
        echo flatsome_apply_shortcode( 'blog_posts', array(
            'columns'      => '3',
            'columns__md'  => '3',
            'columns__sm'  => '2',
            'type'         => get_theme_mod( 'search_result_style', 'slider' ),
            'image_height' => '56.25%',
            'show_date'    => get_theme_mod( 'blog_badge', 1 ) ? 'true' : 'false',
            'ids'          => implode( ',', $posts ),
            'infinitive'   => 'false'
        ) );
    }

    if ( ! empty( $pages ) ) {
        echo '<hr/><h4 class="uppercase">' . esc_html__( 'Pages found', 'erua' ) . ' (' . $pages_count . ')</h4>';
        echo flatsome_apply_shortcode( 'ux_pages', array(
            'columns'      => '3',
            'columns__md'  => '3',
            'columns__sm'  => '2',
            'type'         => get_theme_mod( 'search_result_style', 'slider' ),
            'image_height' => '56.25%',
            'ids'          => implode( ',', $pages ),
            'infinitive'   => 'false'
        ) );
    }

    if ( ! empty( $documents ) ) {
        echo '<hr/><h4 class="uppercase">' . esc_html__( 'Documents found', 'erua' ) . ' (' . $documents_count . ')</h4>';

        // echo '<pre>';
        // var_dump($documents);
        // echo '</pre>';

        $post_ids = implode( ',', $documents );
        echo '<div style="margin-bottom: 40px;">';
        echo do_shortcode("[latest_documents post_ids='{$post_ids}' full_width='true']");
        echo '</div>';

    }

    if ( ! empty( $case_studies ) ) {
        echo '<hr/><h4 class="uppercase">' . esc_html__( 'Case Studies found', 'erua' ) . ' (' . $case_studies_count . ')</h4>';

        // echo '<pre>';
        // var_dump($case_studies);
        // echo '</pre>';

        $post_ids = implode( ',', $case_studies );
        echo '<div style="margin-bottom: 40px;">';
        echo do_shortcode("[latest_studies full_width='true']");
        echo '</div>';

    }

    if ( ! empty( $wfrm_solutions ) ) {
        echo '<hr/><h4 class="uppercase">' . esc_html__( 'WFRM Solutions found', 'erua' ) . ' (' . $wfrm_solutions_count . ')</h4>';

        // echo '<pre>';
        // var_dump($wfrm_solutions);
        // echo '</pre>';

        $post_ids = implode( ',', $wfrm_solutions );
        echo '<div style="margin-bottom: 40px;">';
        echo do_shortcode("[latest_solutions post_ids='{$post_ids}' full_width='true']");
        echo '</div>';

    }

    if ( ! empty( $techmall_solutions ) ) {
        echo '<hr/><h4 class="uppercase">' . esc_html__( 'TechMall Solutions found', 'erua' ) . ' (' . $techmall_solutions_count . ')</h4>';

        // echo '<pre>';
        // var_dump($techmall_solutions);
        // echo '</pre>';

        $post_ids = implode( ',', $techmall_solutions );
        echo '<div style="margin-bottom: 40px;">';
        echo do_shortcode("[latest_solutions post_ids='{$post_ids}' post_type='techmall-solution' full_width='true']");
        echo '</div>';

    }

    ?>
    <?php endif; ?>

    <?php
}
//add_action('woocommerce_after_main_content','erua_pages_in_search_results', 10);
