<?php

if ( ! defined( 'ABSPATH' ) ) {
    exit; // Exit if accessed directly.
}

//================================================================================
// QUERYING
//================================================================================

/**
 * Add new query variables in the list of public query variables, 
 * so that we can use it in our custom URLs.
 * https://codex.wordpress.org/Function_Reference/get_query_var
 */
function add_public_query_vars_for_creative_blog_posts( $qvars ) {
    // Back-end Creative Blog listings
    $qvars[] = "creative_blog_post_meta_query";
    // Front-end Creative Blog listings
    // $myvars = [
    //     "viewing",
    //     "creative_blog_tag",
    //     "creative_blog_category", # Creative Blog Category
    // ];
    // foreach ( $myvars as $key ) {
    //     if ( ! array_key_exists( $key, $qvars ) ) {
    //         $qvars[] = $key;
    //     }
    // }

    return $qvars;
}
add_filter( 'query_vars', 'add_public_query_vars_for_creative_blog_posts' );

/**
 * Manipulate query before quering 
 * and map 'creative_blog_post_meta_query' custom query argument 
 * to the 'meta_query' WordPress's query argument.
 */
function creative_blog_post_pre_get_posts( $query ) {

    // if( function_exists('get_current_screen') && get_current_screen()->parent_base == 'edit' && is_admin() && $query->is_main_query() ) {
    //      //$query->query_vars['suppress_filters'] = true;
    //  }

    /**
     * If we're viewing the administration panel, 
     * and we're quering for "creative-blog-post" custom post type, 
     * and there's a "creative_blog_post_meta_query" custom query argument present, 
     * and its value is not empty.
     */
    if( is_admin() && get_query_var( 'post_type' ) == 'creative-blog-post' && ! empty( get_query_var( 'creative_blog_post_meta_query' ) ) && ! empty( $query->query_vars['creative_blog_post_meta_query'][0]['value'] ) && $query->is_main_query() ) {

        // Map 'creative_blog_post_meta_query' to 'meta_query'
        $query->set( 'meta_query', get_query_var( 'creative_blog_post_meta_query' ) );

    }

}
add_action( 'pre_get_posts', 'creative_blog_post_pre_get_posts' );

//================================================================================
// LISTINGS SCREEN (edit.php) CUSTOMIZATION - COLUMNS, FILTERS etc.
//================================================================================

/**
 * Customize views @edit.php?post_type=creative-blog-post
 * More @ https://codex.wordpress.org/Plugin_API/Filter_Reference/views_edit-post
 */
function remove_views_from_creative_blog_posts($views) {  
    $user = wp_get_current_user();
    if ( in_array( 'administrator', (array) $user->roles ) ) {
        unset($views['mine']);
    }
    return $views;
}
add_filter('views_edit-creative-blog-post', 'remove_views_from_creative_blog_posts');

/**
 * Register columns for the back-end listings of the "creative-blog-post" custom post type.
 * https://codex.wordpress.org/Plugin_API/Filter_Reference/manage_$post_type_posts_columns
 */
function register_backend_creative_blog_post_columns($columns)
{

    $new_fields = [
        'applicant' => __('Applicant', 'erua'),
        'email' => __('Email', 'erua')
    ];
    associative_array_insert_after($columns, 'author', $new_fields);
    unset($columns['author']);

    return $columns;
}
add_filter('manage_creative-blog-post_posts_columns', 'register_backend_creative_blog_post_columns');

/**
 * Create the content of the custom columns
 * that were configured with register_backend_creative_blog_post_columns()
 * http://justintadlock.com/archives/2011/06/27/custom-columns-for-custom-post-types
 * https://wordpress.stackexchange.com/questions/253640/adding-custom-columns-to-custom-post-types
 * https://wpsmackdown.com/easy-filter-posts-wordpress-admin/
 * https://codex.wordpress.org/WordPress_Query_Vars
 * https://wordpress.stackexchange.com/questions/212519/filter-by-custom-field-in-custom-post-type-on-admin-page
 */
function manage_creative_blog_post_backend_custom_columns($column, $post_id)
{
    global $post;

    switch ($column) {

        case 'applicant':

            $author_id = $post->post_author;
            $author = get_the_author_meta('display_name', $author_id);

            if ($author_id) {
                printf('<a href="%s">%s</a>',
                        esc_url(add_query_arg(['post_type' => $post->post_type, 'author' => $author_id], 'edit.php')),
                        $author
                );
            }
            else {
                if ( get_field('field_6566946c973c9', $post_id) && get_field('field_6566946e973ca', $post_id) ) {
                    $applicant_name = get_field('field_6566946c973c9', $post_id);
                    $applicant_email = get_field('field_6566946e973ca', $post_id);

                    printf('<a href="%s">%s</a>',
                        esc_url(add_query_arg(['post_type' => $post->post_type, 'creative_blog_post_meta_query' => [ 
                            'relation' => 'AND', 
                            ['key' => 'creative_blog_applicant_name', 'value' => "$applicant_name", 'compare' => 'LIKE'], 
                            ['key' => 'creative_blog_applicant_email', 'value' => "$applicant_email", 'compare' => 'LIKE']
                        ]], 'edit.php')),
                        $applicant_name . ' (<span style="color: red;" title="' . __('Guest', 'erua') . '">G</span>)'
                    );
                }
                else {
                    echo __('Unknown', 'erua');
                }
            }

            break;

        case 'email':

            $author_id = $post->post_author;
            $author_email = get_the_author_meta('user_email', $author_id);

            if ($author_id) {
                printf('<a href="%s">%s</a>',
                        esc_url(add_query_arg(['post_type' => $post->post_type, 'author' => $author_id], 'edit.php')),
                        $author_email
                );
            }
            else {
                if ( get_field('field_6566946c973c9', $post_id) && get_field('field_6566946e973ca', $post_id) ) {
                    $applicant_name = get_field('field_6566946c973c9', $post_id);
                    $applicant_email = get_field('field_6566946e973ca', $post_id);

                    printf('<a href="%s">%s</a>',
                        esc_url(add_query_arg(['post_type' => $post->post_type, 'creative_blog_post_meta_query' => [ 
                            'relation' => 'AND', 
                            ['key' => 'creative_blog_applicant_name', 'value' => "$applicant_name", 'compare' => 'LIKE'], 
                            ['key' => 'creative_blog_applicant_email', 'value' => "$applicant_email", 'compare' => 'LIKE']
                        ]], 'edit.php')),
                        $applicant_email . ' (<span style="color: red;" title="' . __('Guest', 'erua') . '">G</span>)'
                    );
                }
                else {
                    echo __('Unknown', 'erua');
                }
            }

            break;

        /* Just break out of the switch statement for everything else. */
        default:
            break;
    }
}
add_action('manage_creative-blog-post_posts_custom_column', 'manage_creative_blog_post_backend_custom_columns', 10, 2);

/**
 * Add filters based on custom taxonomies,
 * for the Creative Blog post listings (@ edit.php).
 * https://generatewp.com/filtering-posts-by-taxonomies-in-the-dashboard/
 */
function create_backend_filters_for_creative_blog_posts($post_type, $which)
{

    // Apply this only on a specific post type
    if ('creative-blog-post' === $post_type) {

        /**
         * A list of taxonomy slugs to filter by
         * Note: Private and draft posts will be displayed 
         * but won't be counted in the taxonomy's terms.
         */
        $taxonomies = [
            'creative-blog-category'
        ];

        foreach ($taxonomies as $taxonomy_slug) {

            // Retrieve taxonomy data
            $taxonomy_obj = get_taxonomy($taxonomy_slug);
            $taxonomy_name = $taxonomy_obj->labels->name;
            $taxonomy_is_hierarchical = $taxonomy_obj->hierarchical;

            if ( $taxonomy_is_hierarchical ) {

                /**
                 * Retrieve parent terms
                 */
                $top_level_terms = get_terms( [
                    'taxonomy'      => $taxonomy_slug,
                    'parent'        => '0',
                    'hide_empty'    => false,
                    'suppress_filters' => false
                ] );

                // Display filter HTML
                echo "<select name='{$taxonomy_slug}' id='{$taxonomy_slug}' class='postform'>";
                echo '<option value="">' . sprintf( esc_html__('All %s', 'erua'), $taxonomy_name ) . '</option>';
                foreach ( $top_level_terms as $top_level_term ) {

                    // View all Creative Blog posts with an awaiting nature (pending, waiting for x's reply etc.) of status
                    $query = get_posts(
                        [
                            'post_type' => $post_type,
                            'fields' => 'ids',
                            'post_status' => ['publish', 'pending', 'draft', 'auto-draft', 'future', 'private', 'inherit'],
                            'numberposts' => -1,
                            'tax_query' => [
                                [
                                    'taxonomy' => $taxonomy_slug,
                                    'field' => 'slug',
                                    'terms' => $top_level_term->slug,
                                    'operator' => 'IN'
                                ]
                            ]
                        ]
                    );
                    wp_reset_postdata();
                    $count = count($query);

                    printf(
                        '<option value="%1$s" %2$s>%3$s (%4$s)</option>',
                        $top_level_term->slug,
                        ((isset($_GET[$taxonomy_slug]) && ($_GET[$taxonomy_slug] == $top_level_term->slug)) ? ' selected="selected"' : ''),
                        $top_level_term->name,
                        $count //$top_level_term->count
                    );

                    $top_term_id = $top_level_term->term_id;

                    $second_level_terms = get_terms( array(
                        'taxonomy' => $taxonomy_slug, # you could also use $taxonomy as defined in the first lines
                        'child_of' => $top_term_id,
                        'parent' => $top_term_id, # disable this line to see more child elements (child-child-child-terms)
                        'hide_empty' => false,
                    ) );
                    foreach ($second_level_terms as $second_level_term) {

                        $query = get_posts(
                            [
                                'post_type' => $post_type,
                                'fields' => 'ids',
                                'post_status' => ['publish', 'pending', 'draft', 'auto-draft', 'future', 'private', 'inherit'],
                                'numberposts' => -1,
                                'tax_query' => [
                                    [
                                        'taxonomy' => $taxonomy_slug,
                                        'field' => 'slug',
                                        'terms' => $second_level_term->slug,
                                        'operator' => 'IN'
                                    ]
                                ]
                            ]
                        );
                        wp_reset_postdata();
                        $count = count($query);

                        printf(
                            '<option value="%1$s" %2$s>%3$s (%4$s)</option>',
                            $second_level_term->slug,
                            ((isset($_GET[$taxonomy_slug]) && ($_GET[$taxonomy_slug] == $second_level_term->slug)) ? ' selected="selected"' : ''),
                            '&nbsp;&nbsp;&nbsp;&nbsp;' . $second_level_term->name,
                            $count//$second_level_term->count
                        );
                    }
                }
                echo '</select>';
            }
            else {

                /**
                 * Retrieve parent terms
                 */
                $top_level_terms = get_terms( [
                    'taxonomy'      => $taxonomy_slug,
                    'parent'        => '0',
                    'hide_empty'    => false,
                    'suppress_filters' => false
                ] );

                // Display filter HTML
                echo "<select name='{$taxonomy_slug}' id='{$taxonomy_slug}' class='postform'>";
                echo '<option value="">' . sprintf( esc_html__('Show All %s', 'erua'), $taxonomy_name ) . '</option>';
                foreach ( $top_level_terms as $top_level_term ) {

                    // View all Creative Blog posts with an awaiting nature (pending, waiting for x's reply etc.) of status
                    $query = get_posts(
                        [
                            'post_type' => $post_type,
                            'fields' => 'ids',
                            'post_status' => ['publish', 'pending', 'draft', 'auto-draft', 'future', 'private', 'inherit'],
                            'numberposts' => -1,
                            'tax_query' => [
                                [
                                    'taxonomy' => $taxonomy_slug,
                                    'field' => 'slug',
                                    'terms' => $top_level_term->slug,
                                    'operator' => 'IN'
                                ]
                            ]
                        ]
                    );
                    wp_reset_postdata();
                    $count = count($query);

                    printf(
                        '<option value="%1$s" %2$s>%3$s (%4$s)</option>',
                        $top_level_term->slug,
                        ((isset($_GET[$taxonomy_slug]) && ($_GET[$taxonomy_slug] == $top_level_term->slug)) ? ' selected="selected"' : ''),
                        $top_level_term->name,
                        $count //$top_level_term->count
                    );
                }
                echo '</select>';
            }
        }

    }

}
add_action('restrict_manage_posts', 'create_backend_filters_for_creative_blog_posts', 10, 2);

//================================================================================
// BACK-END FIELD HANDLING
//================================================================================

/**
 * If a post of "creative-blog-post" type was updated,
 * then adjust the "date_updated" custom field.
 */
function creative_blog_post_update_modified_date($post_id, $post, $update)
{
    if ( $post->post_type == 'creative-blog-post' ) {

        if ( $update ) {
            // If the "date_updated" custom field exists, update it.
            if ( get_field('field_65669469973c8') ) {
                update_field('field_65669469973c8', get_the_modified_time('j F Y, H:i', $post_id), $post_id);
            }
        }
        else if ( !$update ) {
            // If the "date_created" custom field value does not exist, initialiaze it.
            if ( !get_field('field_65669467973c7') ) {
                update_field( 'field_65669467973c7', get_post_time('j F Y, H:i', false, $post_id, false ), $post_id );
            }
            // If the "date_updated" custom field value does not exist, initialiaze it.
            if ( !get_field('field_65669469973c8') ) {
                update_field( 'field_65669469973c8', get_post_time('j F Y, H:i', false, $post_id, false ), $post_id );
            }
        }

    }
}
add_action('save_post', 'creative_blog_post_update_modified_date', 10, 3);

function creative_blog_post_load_applicant_name( $value, $post_id, $field ) {

    $user_id = get_field('creative_blog_registered_user_id', $post_id);

    if ( !$user_id ) {
        return $value;
    }

    $user = get_user_by('id', (int) $user_id);

    if ( $value != $user->display_name ) {
        $value = $user->display_name;
    }

    return $value;
}
add_filter('acf/load_value/key=field_6566946c973c9', 'creative_blog_post_load_applicant_name', 10, 3); # Applicant's Name

function creative_blog_post_load_applicant_email( $value, $post_id, $field ) {

    $user_id = get_field('creative_blog_registered_user_id', $post_id);

    if ( !$user_id ) {
        return $value;
    }

    $user = get_user_by('id', (int) $user_id);

    if ( $value != $user->user_email ) {
        $value = $user->user_email;
    }

    return $value;
}
add_filter('acf/load_value/key=field_6566946e973ca', 'creative_blog_post_load_applicant_email', 10, 3); # Applicant's E-mail

function creative_blog_post_load_applicant_profile_url( $value, $post_id, $field ) {

    if ( function_exists('bp_is_active') ) {

        $user_id = get_field('creative_blog_registered_user_id', $post_id);

        if ( !$user_id ) {
            return $value;
        }

        $user_profile_url = bbp_get_user_profile_url( (int) $user_id );

        if ( $value != $user_profile_url ) {
            $value = $user_profile_url;
        }
    }

    return $value;
}
add_filter('acf/load_value/key=field_65669474973cc', 'creative_blog_post_load_applicant_profile_url', 10, 3); # Applicant's Profile URL

function creative_blog_post_render_applicant_profile_url( $field ) {
    // echo '<pre>';
    // var_dump($field);
    // echo '</pre>';
    if ( $field['value'] ) {
        echo '<a href="'.$field['value'].'" target="_blank" style="display: inline-block; background-color: aliceblue; padding: 5px; border-radius: 0 0 5px 5px; text-decoration: none;">'.__('Visit profile', 'erua').'</a>';
    }
}
add_action( 'acf/render_field/key=field_65669474973cc', 'creative_blog_post_render_applicant_profile_url', 10, 1 );

//================================================================================
// DISABLING STUFF
//================================================================================

/**
 * Make an ACF field readonly to prevent editing.
 * https://www.advancedcustomfields.com/resources/acf-load_field/
 * https://support.advancedcustomfields.com/forums/topic/read-only-field-2/
 */
function creative_blog_post_acf_make_field_readonly( $field ) {

    // $field['choices'] = array(
    //     'custom' => 'My Custom Choice'
    // );
    //$field['disabled'] = 1;
    $field['readonly'] = true;

    return $field;

}
/**
 * Make an ACF field disabled to prevent editing.
 * https://www.advancedcustomfields.com/resources/acf-load_field/
 * https://github.com/AdvancedCustomFields/acf/issues/26
 */
function creative_blog_post_acf_make_field_disabled( $field ) {

    $field['disabled'] = true;

    return $field;

}
add_filter('acf/load_field/key=field_6566946c973c9', 'creative_blog_post_acf_make_field_readonly'); # Applicant's Name
add_filter('acf/load_field/key=field_65669471973cb', 'creative_blog_post_acf_make_field_readonly'); # Applicant's User ID
add_filter('acf/load_field/key=field_65669467973c7', 'creative_blog_post_acf_make_field_readonly'); # Date created
add_filter('acf/load_field/key=field_65669469973c8', 'creative_blog_post_acf_make_field_readonly'); # Date updated
add_filter('acf/load_field/key=field_6566990e4a798', 'creative_blog_post_acf_make_field_readonly'); # Unique Token
if ( !current_user_can('administrator') ) {
    add_filter('acf/load_field/key=field_6566946e973ca', 'creative_blog_post_acf_make_field_readonly'); # Applicant's E-mail
}

/**
 * Disable various fields at a Post's editing screen.
 */
function erua_disable_creative_blog_post_fields() {
    global $pagenow;

    // If we are at the back-office, at a post.
    if ( is_admin() && !( defined( 'DOING_AJAX' ) && DOING_AJAX ) && ($pagenow == 'post.php' || $pagenow == 'post-new.php') && get_post_type() == 'creative-blog-post' ) {
    ?>
        <script>
        jQuery(document).ready(function() {
            // Disable ACF's date picker at Post's editing screen.
            jQuery('#creative-blog-date-created .hasDatepicker').prop('disabled', 'disabled');
            jQuery('#creative-blog-date-updated .hasDatepicker').prop('disabled', 'disabled');

            jQuery("div[data-key='field_65669467973c7']").css({position: "relative"});
            jQuery("div[data-key='field_65669467973c7']").prepend('<div id="date-created-tracking-jammer" style="display:block; position:absolute; top:0; left:0; width:100%; height:100%; cursor: initial; z-index:9; background-color: aliceblue;"></div>');
            jQuery("div[data-key='field_65669469973c8']").css({position: "relative"});
            jQuery("div[data-key='field_65669469973c8']").prepend('<div id="date-updated-tracking-jammer" style="display:block; position:absolute; top:0; left:0; width:100%; height:100%; cursor: initial; z-index:9; background-color: aliceblue;"></div>');

            // Disable various Publishing actions
            jQuery('body.wp-admin #misc-publishing-actions .edit-timestamp').css({display: "none"});

            <?php //if ( is_user_logged_in() && !current_user_can('administrator') ) : ?>
            // Disable Title
            //jQuery('#title').prop('disabled', true);
            // Disable various Publishing actions
            jQuery('body.wp-admin #misc-publishing-actions #major-publishing-actions').css({display: "none"});

            // Hide Status Tracking
            jQuery("div[data-key='field_65668d4010fb9']").css({position: "relative"});
            jQuery("div[data-key='field_65668d4010fb9']").prepend('<div id="status-tracking-jammer" style="display:block; position:absolute; top:0; left:0; width:100%; height:100%; cursor: initial; z-index:9; background-color: aliceblue;"></div>');

            // Hide Unique ID
            jQuery("div[data-key='field_6566990e4a798']").css({position: "relative"});
            jQuery("div[data-key='field_6566990e4a798']").prepend('<div id="unique-id-jammer" style="display:block; position:absolute; top:0; left:0; width:100%; height:100%; cursor: initial; z-index:9; background-color: aliceblue;"></div>');

            <?php //endif; ?>
        });
        </script>
    <?php
    }

}
add_action('admin_print_footer_scripts', 'erua_disable_creative_blog_post_fields');

//================================================================================
// BUBBLE NOTIFICATIONS
//================================================================================

/**
 * Notification "bubble" with pending/open Posts count
 */
add_action('admin_menu', function () {
    global $menu;
    //$count_posts = wp_count_posts('creative-blog-post');
    //$count = $count_posts->pending;

    // View all Posts with an awaiting nature (pending, waiting for x's reply etc.) of status
    $creative_blog_posts_query = get_posts(
        [
            'post_type' => 'creative-blog-post',
            'fields' => 'ids',
            //'post_status' => 'any', // Published, Pending etc.
            //'post_status' => ['publish', 'pending', 'draft', 'auto-draft', 'future', 'private', 'inherit', 'trash'],
            //'post_status' => ['publish', 'pending', 'draft', 'future', 'private', 'inherit'],
            'post_status' => ['pending', 'draft'],
            'numberposts' => -1,
        ]
    );
    if (!empty($creative_blog_posts_query)) wp_reset_postdata();
    $count = count($creative_blog_posts_query);
    /* echo '<pre>';
    var_dump($creative_blog_posts_query);
    echo '</pre>';
    die(); */

    $menu_item = wp_list_filter(
        $menu,
        [2 => 'edit.php?post_type=creative-blog-post'] // 2 is the position of an array item which contains URL, it will always be 2!
    );
    if ( !empty($menu_item) && $count > 0 ) {
        $menu_item_position = key($menu_item); // get the array key (position) of the element
        $menu[$menu_item_position][0] .= ' <span class="awaiting-mod" title="' . $count . ' Posts require actions">' . $count . '</span>';
    }
});
