<?php

if ( ! defined( 'ABSPATH' ) ) {
    exit; // Exit if accessed directly.
}

//================================================================================
// ADVANCED CUSTOM FIELDS (ACF) - SAVING/LOADING FROM FILESYSTEM
//================================================================================

/**
 * - Custom saving point for acf fields/groups -
 * 
 * Each time we save a field group a JSON file will be created (or updated) 
 * with the field group and field settings. The JSON file will be named using the field group’s unique key.
 * 
 * https://www.advancedcustomfields.com/resources/local-json/
 */
function my_acf_json_save_point( $path ) {

    // update path
    $path = get_stylesheet_directory() . '/assets/acf-json/';

    // return
    return $path;

}
add_filter('acf/settings/save_json', 'my_acf_json_save_point');

/**
 * - Custom loading point of acf fields/groups -
 * 
 * ACF will load the relevant field group and field settings from this file 
 * which reduces the number of database calls during page load.
 */
function my_acf_json_load_point( $paths ) {

    // remove original path (optional)
    unset($paths[0]);

    // append path
    $paths[] = get_stylesheet_directory() . '/assets/acf-json/';

    // return
    return $paths;
    
}
add_filter('acf/settings/load_json', 'my_acf_json_load_point');

//================================================================================
// FIELD GROUP LOCATION CUSTOMIZATIONS
//================================================================================

/**
 * Location = "Nowhere"
 * https://support.advancedcustomfields.com/forums/topic/location-nowhere/
 */
function acf_location_rules_values_post_type($choices) {
  $choices['none'] = 'NO LOCATION';
  return $choices;
}
add_filter('acf/location/rule_values/post_type', 'acf_location_rules_values_post_type');

//================================================================================
// FORM HOOKING TO ENDPOINTS
//================================================================================

/**
 * Add AFC form head before get_header()
 *
 * https://www.advancedcustomfields.com/resources/acf_form_head/
 * https://stackoverflow.com/questions/67270667/wordpress-page-goes-blank-after-form-submit
 * https://stackoverflow.com/questions/38444557/detect-dashboard-of-woocommerce-my-account-pages
 */
function erua_add_acf_form_head_to_page() {
  global $post;

  // Add ACF form head to the "Events" page
  if ( is_user_logged_in() && is_page() && get_field('acf_form_head_enabled', $post->ID) ) {
      acf_form_head();
  }

}
add_action( 'get_header', 'erua_add_acf_form_head_to_page', 0 );

//================================================================================
// FIELD CUSTOMIZATIONS
//================================================================================

/**
 * Add height field to ACF WYSIWYG,
 * at the back-office.
 * 
 * https://gist.github.com/stianandreassen/6dc87c88c43b2bc43d0ea1a94bd5cd1e
 */
function my_acf_form_wysiwyg_render_field_settings( $field ) {
	acf_render_field_setting( $field, array(
		'label'			=> __('Height of Editor'),
		'instructions'	=> __('Height of Editor after Init'),
		'name'			=> 'wysiwyg_height',
		'type'			=> 'number',
	));
}
//add_action('acf/render_field_settings/type=wysiwyg', 'my_acf_form_wysiwyg_render_field_settings', 10, 1 );

/**
 * Render height on ACF WYSIWYG 
 */
function my_acf_form_wysiwyg_render_field( $field ) {
	$field_class = '.acf-'.str_replace('_', '-', $field['key']);
?>
	<style type="text/css">
	<?php echo $field_class; ?> iframe {
		min-height: <?php echo $field['wysiwyg_height']; ?>px; 
	}
	</style>
	<script type="text/javascript">
	jQuery(window).load(function() {
		jQuery('<?php echo $field_class; ?>').each(function() {
			jQuery('#'+jQuery(this).find('iframe').attr('id')).height( <?php echo $field['wysiwyg_height']; ?> );
		});
	});
	</script>
<?php
}
//add_action( 'acf/render_field/type=wysiwyg', 'my_acf_form_wysiwyg_render_field', 10, 1 );

/**
 * Add/remove toolbars from the WYSIWYG editor.
 * https://support.advancedcustomfields.com/forums/topic/apply-filters-to-wysiwyg/
 */
function my_acf_form_wysiwyg_modify_heightacf_form_wysiwyg_modify_toolbars( $toolbars ) {
	$toolbars['Full'] = array();
	$toolbars['Full'][1] = array('bold', 'italic', 'underline', 'bullist', 'numlist', 'alignleft', 'aligncenter', 'alignright', 'alignjustify', 'link', 'unlink', 'hr', 'spellchecker', 'wp_more', 'wp_adv' );
	$toolbars['Full'][2] = array('styleselect', 'formatselect', 'fontselect', 'fontsizeselect', 'forecolor', 'pastetext', 'removeformat', 'charmap', 'outdent', 'indent', 'undo', 'redo', 'wp_help' );

	// remove the 'Basic' toolbar completely (if you want)
	unset( $toolbars['Basic' ] );

	// return $toolbars - IMPORTANT!
	return $toolbars;
}
//add_filter('acf/fields/wysiwyg/toolbars' , 'my_acf_form_wysiwyg_modify_heightacf_form_wysiwyg_modify_toolbars');

// Other: https://stackoverflow.com/questions/51796749/wp-advanced-custom-fields-wysiwyg-insert-link-to-media

/**
 * Set height of WYSIWYG editor on render.
 * 
 * https://gist.github.com/courtneymyers/eb51f918181746181871f7ae516b428b#gistcomment-3631773
 */
function my_acf_form_wysiwyg_modify_height() {
    ?>
      <style>
        .acf-editor-wrap iframe {
          min-height: 0;
        }
      </style>
      <script>
        (function($) {
          // reduce placeholder textarea height to match tinymce settings (when using delay-setting)
          $('.acf-editor-wrap.delay textarea').css('height', '100px');
          // (filter called before the tinyMCE instance is created)
          acf.add_filter('wysiwyg_tinymce_settings', function(mceInit, id, $field) {
            // enable autoresizing of the WYSIWYG editor
            mceInit.wp_autoresize_on = true;
            return mceInit;
          });
          // (action called when a WYSIWYG tinymce element has been initialized)
          acf.add_action('wysiwyg_tinymce_init', function(ed, id, mceInit, $field) {
            // reduce tinymce's min-height settings
            ed.settings.autoresize_min_height = 100;
            // reduce iframe's 'height' style to match tinymce settings
            $('.acf-editor-wrap iframe').css('height', '100px');
          });
        })(jQuery)
      </script>
    <?php
}
add_action('acf/input/admin_footer', 'my_acf_form_wysiwyg_modify_height');

/**
 * Add Text Color to WYSIWYG editor.
 * https://support.advancedcustomfields.com/forums/topic/how-do-you-add-text-color-button-to-tinymce-basic-toolbar/
 */
function my_acf_form_wysiwyg_add_text_color_to_toolbar( $toolbars ) {
	array_unshift( $toolbars['Basic' ][1], 'forecolor' );
  //array_unshift( $toolbars['Full' ][1], 'forecolor' );
	return $toolbars;
}
add_filter( 'acf/fields/wysiwyg/toolbars' , 'my_acf_form_wysiwyg_add_text_color_to_toolbar'  );

/**
 * Add some custom colors to WYSIWYG editors.
 * 
 * https://wordpress.stackexchange.com/questions/233450/how-do-you-add-custom-color-swatches-to-all-wysiwyg-editors
 */
function my_wysiwyg_custom_colors($init) {

    $custom_colours = '
        "3366FF", "Color 1 name",
        "CCFFCC", "Color 22 name hmm",
        "FFFF00", "Color 3 name",
        "99CC00", "Color 4 name",
        "FF0000", "Color 5 name",
        "FF99CC", "Color 6 name",
        "CCFFFF", "Color 7 name"
    ';

    // build colour grid default+custom colors
    $init['textcolor_map'] = '['.$custom_colours.']';

    // change the number of rows in the grid if the number of colors changes
    // 8 swatches per row
    $init['textcolor_rows'] = 8;

    return $init;
}
//add_filter('tiny_mce_before_init', 'my_wysiwyg_custom_colors');

/**
 * Hex to RGB color conversion.
 * Change return value from hexadecimal to RGB.
 * https://github.com/elliotcondon/acf/issues/753
 * https://support.advancedcustomfields.com/forums/topic/color-picker-values/
 */
function my_acf_format_value( $value, $post_id, $field ) {

	if ( $value[0] == '#' ) {
		$value = substr( $value, 1 );
	}
	if ( strlen( $value ) == 6 ) {
		list( $r, $g, $b ) = [ $value[0] . $value[1], $value[2] . $value[3], $value[4] . $value[5] ];
	} elseif ( strlen( $value ) == 3 ) {
		list( $r, $g, $b ) = [ $value[0] . $value[0], $value[1] . $value[1], $value[2] . $value[2] ];
	} else {
		return false;
	}
	$r = hexdec( $r );
	$g = hexdec( $g );
	$b = hexdec( $b );
	
	//return [ 'red' => $r, 'green' => $g, 'blue' => $b ];
    //       ===or===
    //return 'rgb(' . $r . ', ' . $g . ', ' . $b . ')';
    return $r . ', ' . $g . ', ' . $b;
}
//add_filter('acf/format_value/type=color_picker', 'my_acf_format_value', 10, 3);
